package easik.sketch.constraint;

import java.util.ArrayList;

import easik.sketch.edge.SketchEdge;
import easik.sketch.path.SketchPath;


/** 
 *	The commutative diagram is a constraint made up of two lists (Arrays) of 
 *	sketch edges. These two lists should share a domain and codomain.	
 *
 * @author Kevin Green 2006
 * @author Vera Ranieri 2006
 * @version 2006-06-13 Kevin Green
 */
public class CommutativeDiagram extends Constraint {

	/**
	 * Creates a commutative diagram from an ArrayList of paths.
	 * The ArrayList should contain only two paths.
	 * Also sets details of the visual representation.
	 * 
	 * @param inPaths ArrayList that should contain only the two paths used for the commutative diagram
	 * @param x x coordinate
	 * @param y y coordinate
	 * @param isVisible If true the constraint will be shown in the graph, otherwise it will not be visible.
	 */
	public CommutativeDiagram(ArrayList<SketchPath> inPaths, int x, int y, boolean isVisible) {
		super("CD", x, y, isVisible);
		_paths = inPaths;
		
		addEdges();
		
		_constraintLabel = "CD";
	}
	
	/**
	 * Creates a commutative diagram from two paths.
	 * 
	 * @param pathA The first path.
	 * @param pathB The second path.
	 */
	public CommutativeDiagram(SketchPath pathA, SketchPath pathB) {
		_paths = new ArrayList<SketchPath>();
		_paths.add(pathA);
		_paths.add(pathB);
		
		addEdges();
		
		_isVisible = true;
		_constraintLabel = "CD";
	}
	
	/**
	 * Creates a commutative diagram from two path.
	 * Also sets details of the visual representation.
	 * 
	 * @param pathA The first path
	 * @param pathB The second path
	 * @param x x coordinate
	 * @param y y coordinate
	 * @param isVisible If true the constraint will be shown in the graph, otherwise it will not be visible.
	 */
	public CommutativeDiagram(SketchPath pathA, SketchPath pathB, int x, int y, boolean isVisible) {
		super("CD", x, y, isVisible);
		_paths = new ArrayList<SketchPath>();
		_paths.add(pathA);
		_paths.add(pathB);
		addEdges();
		_constraintLabel = "CD";
	}

	/**
	 * Return the first path
	 * 
	 * @return The first path
	 */
	public SketchPath getPathA() {
		return (SketchPath) _paths.get(0);
	}

	/**
	 * Return the second path
	 * 
	 * @return The second path
	 */
	public SketchPath getPathB() {
		return (SketchPath) _paths.get(1);
	}
	
	/**
	 * Checks whether the current selected paths form a commutative diagram.
	 * This is determined by checking whether the two paths participating in the commutative
	 * diagram have the same domain and codomain.
	 * 
	 * @param pathA The first path participating in the commutative diagram
	 * @param pathB the second path participating in the commutative diagram
	 * @return True if it forms a commutative diagram, false otherwise
	 * @since 2006-05-25 Vera Ranieri
	 */
	public static boolean isCommutativeDiagram(SketchPath pathA, SketchPath pathB){
		
		if(!(pathA.getDomain().equals(pathB.getDomain()) && 
				pathA.getCoDomain().equals(pathB.getCoDomain()))){
			return false;
		}
		if(pathA.getEdges().size() == pathB.getEdges().size()){
			boolean different = false;
			int size = pathA.getEdges().size();
			for(int i = 0; i< size; i++){
				if(!pathA.getEdges().get(i).equals(pathB.getEdges().get(i))){
					different = true;
				}
			}
			return different;
		}
		
		return true;
	}

}
